function [Flags, Tonal_list, Non_tonal_list] = ...
   Find_tonal_components(X, TH, Map, CB) % assume fs = 44100 fs
%  [Flags, Tonal_list, Non_tonal_list] = Find_tonal_components(X, TH, Map, CB)
%
%   Identifica y lista tanto las componentes tonales como las no tonales de
%   una señal de audio. Asume en esta implementación que la frecuencia de
%   muestreo es  44100 Hz. Details bare given in [1, pp. 112].  
%
%   See also Decimation
   
%   Author: Fabien A.P. Petitcolas
%           Computer Laboratory
%           University of Cambridge
%   Copyright (c) 1998 by Fabien A.P. Petitcolaso
%   $Id: Find_tonal_components.m,v 1.4 1998-07-07 14:39:54+01 fapp2 Exp $

%   References:
%    [1] Information technology -- Coding of moving pictures and associated
%        audio for digital storage media at up to 1,5 Mbits/s -- Part3: audio.
%        British standard. BSI, London. October 1993. Implementation of ISO/IEC
%        11172-3:1993. BSI, London. First edition 1993-08-01.
%
%   Legal notice:
%    This computer program is based on ISO/IEC 11172-3:1993, Information
%    technology -- Coding of moving pictures and associated audio for digital
%    storage media at up to about 1,5 Mbit/s -- Part 3: Audio, with the
%    permission of ISO. Copies of this standards can be purchased from the
%    British Standards Institution, 389 Chiswick High Road, GB-London W4 4AL, 
%    Telephone:+ 44 181 996 90 00, Telefax:+ 44 181 996 74 00 or from ISO,
%    postal box 56, CH-1211 Geneva 20, Telephone +41 22 749 0111, Telefax
%    +4122 734 1079. Copyright remains with ISO.
%-------------------------------------------------------------------------------
Common;

% Comprueba longitud correcta de la señal X
if (length(X) ~= FFT_SIZE)
   error('Tamaño de Densidad de potencia espectral no esperado');
end

% Crea vector base muestras
if (DRAW),
   t = 1:length(X);
end

% List of flags for all the frequency lines (1 to FFT_SIZE / 2)
Flags = zeros(FFT_SIZE / 2, 1) + NOT_EXAMINED;

% Etiqueta los máximos locales
local_max_list = [];
counter = 1;
for k = 2:FFT_SIZE / 2 -  1,                                     % No nos preocupan los bordes
   if (X(k) > X(k-1) & X(k) >= X(k+1) & k > 2 & k <= 250)  % Busca máximos locales
      local_max_list(counter, INDEX) = k;
      local_max_list(counter, SPL) = X(k);
      counter = counter + 1;
   end
end
% En la variabel local_max_list
% La primera columna tiene el índice de la muestra donde ocurren, la
% segunda columna contiene el valor del máximo.
if (DRAW),
    figure(8)
   clg
   disp('Máximos locales.');
   plot(t, X(t), local_max_list(:, INDEX), local_max_list(:, SPL), 'bo');
   xlabel('Índice de frecuencia'); ylabel('dB'); title('Máximo local.');
   axis([0 256 0 100]); pause;
end

s='lista los maximos locales'
local_max_list
pause

% Lista las componentes tonales y calcula el nivel de presión sonora
Tonal_list = [];
counter = 1;
if not(isempty(local_max_list))
	for i = 1:length(local_max_list(:, 1)),
	   k = local_max_list(i, INDEX);
	   is_tonal = 1;
	   
	   % Layer I
	   % Examina las frecuencias cercanas
	   if (2 < k & k < 63)
	      J = [-2 2];
	   elseif (63 <= k & k < 127)
	      J = [-3 -2 2 3];
	   elseif (127 <= k & k < 250)
	      J = [-6:-2, 2:6];
	   else
	      is_tonal = 0;
       end
	   
       % En este bucle se comprueba si la diferencia entre un maximo local
       % y el situado a distancia j es mayor de 7, lo que indicaría que es
       % tonal (mucha diferencia de amplitud entre uno y otro)
	   for j = J,
	      is_tonal = is_tonal & (X(k) - X(k + j) >= 7);
       end
	   
	   % Si X(k) es una componente tonal ahora, entonces se listan lo
	   % siguiente:
	   %    - Índice  k de la línea espectral
	   %    - Nivel de presión sonora
	   %    - Activa la bandera tonal
	   if is_tonal
	      Tonal_list(counter, INDEX) = k;
	      Tonal_list(counter, SPL) = 10 * log10(10^(X(k - 1) / 10) + ...
	         10^(X(k) / 10) + 10^(X(k + 1) / 10));
	      Flags(k) = TONAL;
	      for j = [J -1 1],
	         Flags(k + j) = IRRELEVANT;
	      end
	      counter = counter + 1;
	   end
    end
    s='lista de componentes tonales'
   Tonal_list
    pause
   
	if (DRAW),
        figure(9)
        clg
	   disp('Componentes Tonales');
	   plot(t, X(t), 'k',Tonal_list(:, INDEX), Tonal_list(:, SPL), 'r*');
	      xlabel('Índice de frecuencia'); ylabel('dB');  title('Componentes Tonales');
	   axis([0 256 0 100]); pause;
   end
end
% Lista las componentes no tonales y calcula su potencia
% Todas las líneas espectrales que no ha se han examinado durante la
% búsqueda previa se suman juntas para formar las componentes no tonales. 
Non_tonal_list = [];
counter = 1;
for i = 1:length(CB(:, 1)) - 1,,
   % Para cada banda crítica calcula la potencia de las componentes no
   % tonales 
   power  = MIN_POWER; % Suma parcial
   weight = 0; %  Utilizado para calcular la media geométrica de las bandas críticas
   for k = TH(CB(i), INDEX):TH(CB(i + 1), INDEX) - 1, % En cada banda crítica
      if (Flags(k) == NOT_EXAMINED),
         power    = 10 * log10(10^(power / 10) + 10^(X(k) / 10));
         weight   = weight + 10^(X(k) / 10) * (TH(Map(k), BARK) - i);
         Flags(k) = IRRELEVANT;
      end
   end
   
   % El índice para las componentes no tonales es el índice más cercano a
   % la media geométrica de la banda crítica. 
   if (power <= MIN_POWER)
      index = round(mean(TH(CB(i), INDEX) + TH(CB(i + 1), INDEX)));
   else
      index = TH(CB(i), INDEX) + round(weight / 10^(power / 10) * ...
         (TH(CB(i + 1), INDEX) - TH(CB(i), INDEX)));
   end
   if (index < 1)
      index = 1;
   end
   if (index > length(Flags))
      index = length(Flags);
   end
   if (Flags(index) == TONAL)
      index = index + 1; % Dos componentes tonales no pueden ser consecutivas.
   end
   
   % Para cada subbanda 
   %   - Índice de las componentes no tonales
   %   - Nivel de presión sonora de esta componente
   Non_tonal_list(i, INDEX) = index;
   Non_tonal_list(i, SPL) = power;
   Flags(index) = NON_TONAL;
end
s='lista de componentes no tonales'
Non_tonal_list
pause

if (DRAW),
    figure(10)
   clg
   disp('Componentes Tonales y no tonales');
   plot(t, X(t),'k', Tonal_list(:, INDEX), Tonal_list(:, SPL), 'r*', ...
      Non_tonal_list(:, INDEX), Non_tonal_list(:, SPL), 'b*');
	      xlabel('Índice de frecuencia'); ylabel('dB');  title('Componentes Tonales y no tonales');
   axis([0 256 0 100]); pause;
end
